	function [h, Amin] = LTH_BAND_FIR(L, N, wcT, wsT, dc, ds)
	
	%	Program for designing, using linear programming,
	%	Lth-band or Nyquist FIR filters. Optimizes the 
	%	coefficients in the minimax sense.
	
	%	Toolbox for DIGITAL FILTERS USING MATLAB
 	
	% 	Author: 		Tapio Saramki
	% 	Modified by: 	LW 2015-09-09	
	%	Copyright 1998 	Tapio Saramki
	% 					Tampere University of Technology.
	% 	Version:			 
	% 	Known bugs:		 
	% 	Report bugs to:	tapio.saramaki@tut.fi	
	
	rho = 1-L*wcT/pi;
	%---------------------------------------------------------------
	N2 = N/2+1; % Half the filter order+1 (number of unknown filter coefficients)
	%---------------------------------
	gridp = floor(30*N2/L); % gridp and grids are the number of points in 
	grids = floor(30*N2*(L-1)/L); % the passband and stopband, respectively
	%--------------------------------------------------------
	Des_pass = ones(1,gridp); % Desired response in the passband and stopband
	Des_stop = zeros(1,grids);
	%--------------------------------------------------------
	relw = dc/ds;	% Passband and stopband weights
	Weig_pass = ones(1,gridp)/relw;
	Weig_stop = ones(1,grids);
	%*******************************************************
	% Set up matrises for Linear Programming:
	% min z = c'x subject to Ax <= b
	% Note that a linear combination of unknowns is minimized. 
	%*******************************************************
	% For our problem, x(i) = h(N2-i+1) for i = 1,2,...,N2, where
	% N2 = N/2+1 with N being the filter order. x(N2+1) is ds,
	% the maximum absolute value of the weighted error function
	% E(omega) = W(omega)[H(omega)-D(omega)], where H(omega) = 
	% x(1)*Phi(1,Omega)+...+x(N)*Phi(N,omega) with
	% Phi(1,Omega) = 1 and Phi(i,Omega) =  2*cos((i-1)*omega)
	% for i > 1.
	%*******************************************************
	W = [0:wcT/(gridp-1):wcT]; 			% Passband grid points
	W = [W wsT:(pi-wsT)/(grids-1):pi];	% Add the stopband grid points
	%************************************************************************
	% Frequency-domain conditions
	% H(omega(i))-delta_s/W(omega(i)) < =  D((omega(i)) and
	% -H(omega(i))-delta_s/W(omega(i)) < =  -D((omega(i))
	%**************************************************************************
	for i = 1:N2
		G(:,i) = TYPE1FRE(i,W)';
	end
	A1 = -1./[Weig_pass Weig_stop]';
	A = [G A1; -G A1];
	b = [Des_pass Des_stop -Des_pass -Des_stop]';
	%********************************************
	% To be minimized is x(N2+1)
	%********************************************
	c = [zeros(1,N2) 1];
	%***************************************************************************
	% Time-domain conditions: x(1) = 1/L and x(1+rL) = 0 for r = 1,2,..., floor(N2/L).
	%***************************************************************************
	clear A1
	clear b1
	clear A2
	clear b2
	%-----------
	% center tap
	%------------
	b1 = [1/L]'; b2 = [-1/L]';
	A1 = [1 zeros(1,N2)]; A2 = [-1 zeros(1,N2)];
	b = [b;b1;b2]; A = [A;A1;A2];
	%-------------------
	% Zero-valued taps
	%-------------------
	for j = L+1:L:N2
		clear A1
		clear b1
		clear A2
		clear b2
		b1 = [0]'; b2 = [0]';
		A1 = [zeros(1,j-1) 1 zeros(1,N2-j+1)];
		A2 = [zeros(1,j-1) -1 zeros(1,N2-j+1)];
		b = [b;b1;b2]; A = [A;A1;A2];
	end
	%**************************************************
	% Call linprog function for Linear Programming
	% This function is included in Optimization Toolbox
	%**************************************************
	New = 0;
	if New == 1
		x = linprog(c, A,b);
	else
		x = lp(c,A,b);
	end
	%-------------------------------------------
	% h(N-i+1) = x(i) for i = 1,2,...N2 (N2 = N/2+1)
	% x(N2+1) is the stopband ripple.
	%-------------------------------------------
	B = x(1:N2)';
	h = [fliplr(x(2:N2)') B]; % The impulse response
	Amin = -H_2_MAG(x(N2+1));
	%========
	function ares = TYPE1FRE(i,W)
	%
	%	Calculates 2*cos((i-1)*W) for i > 1 and
	%	ones(size(W)) for i = 1.
	%
	%	Toolbox for DIGITAL FILTERS USING MATLAB
	
	% 	Author: 		Copyright Tapio Saramaki, 1998
	% 					Tampere University of Technology.
	% 	Modified by: 	LW 2015-09-15
	% 	Known bugs:		
	% 	Report bugs to:	lars.wanhammar@liu.se	
	
	if i > 1
		ares = 2*cos((i-1)*W);
	else
		ares = ones(size(W));
	end
